#ifndef __CMenuItem__
#define __CMenuItem__

#include <Basics/CString.hpp>
#include <Basics/CCountedObject.hpp>
#include "../Graphics/IImage.hpp"
#include "../Basics/CAlphaColour.hpp"

//	===========================================================================

using Exponent::Basics::CString;
using Exponent::Basics::CCountedObject;
using Exponent::GUI::Graphics::IImage;
using Exponent::GUI::Basics::CAlphaColour;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Controls
		{
			/**
			 * @class CMenuItem CMenuItem.hpp
			 * @brief A representation of a menu item. Each menu item can have an image, a string, a sub menu item. You can also disable the item
			 *
			 * @date 12/04/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CMenuItem.hpp,v 1.8 2007/02/08 21:07:54 paul Exp $
			 */
			class CMenuItem : public CCountedObject
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================

			public:

//	===========================================================================

				/**
				 * @enum EMenuItemTypes
				 * @brief Types of menu item
				 */
				enum EMenuItemTypes
				{
					e_menuItemNormal = 0,		/**< A simple string -> possibly with an icon */
					e_menuItemHtmlLink,			/**< A string that links to a web page */
					e_menuItemSeperator,		/**< A serperator -> a full line accross the menu window */
					e_menuItemSelectable,		/**< Item is an on off button */
					e_subMenuHolder,			/**< Holds sub menus */
					e_verticalSeperator,		/**< Vertical seperating bar */
				};

//	===========================================================================
				
				/**
				 * Construction
				 * @param text The string to display
				 * @param identifier The identifier to use
				 * @param type The type of menu item
				 * @param enabled If false menuis disabled, else enabled
				 */
				CMenuItem(const CString &text, const long identifier, const EMenuItemTypes type = e_menuItemNormal, const bool enabled = true);

				/**
				 * Destruction
				 */
				virtual ~CMenuItem();

//	===========================================================================
				
				/**
				 * Set the text
				 * @param text The string to display
				 */
				virtual void setText(const CString &text) { m_text = text; }
				
				/**
				 * Set the id
				 * @param id The identifier to use
				 */
				virtual void setId(const long id) { m_id = id; }
				
				/**
				 * Set the type
				 * @param type The type of menu item
				 */
				virtual void setType(const EMenuItemTypes type) { m_type = type; }
				
				/**
				 * Enable the unit
				 * @param enabled If false menuis disabled, else enabled
				 */
				virtual void enable(const bool enabled = true) { m_isEnabled = enabled; }
				
				/**
				 * Set the index
				 * @param index The local menu index
				 */
				virtual void setIndex(const long index) { m_index = index; }

				/**
				 * Set the images
				 * @param normalImage The icon to use when mouse is not over or down
				 * @param rolloverImage THe icon to use when mouse is over or down
				 */
				virtual void setImages(IImage *normalImage, IImage *rolloverImage)
				{
					EXCHANGE_COUNTED_OBJECTS(m_normalImage, normalImage);
					EXCHANGE_COUNTED_OBJECTS(m_rolloverImage, rolloverImage);
				}

//	===========================================================================
				
				/**
				 * Get the text
				 * @retval const CString& The string being displayed
				 */
				virtual const CString &getText() const { return m_text; }
				
				/**
				 * Get the id
				 * @retval long The menu unique id
				 */
				virtual long getId() const { return m_id; }
				
				/**
				 * Get the type
				 * @retval EMenuItemTypes The type of menu item
				 */
				virtual EMenuItemTypes getType() const { return m_type; }
				
				/**
				 * is the menu item enabled?
				 * @retval bool True if menu is enabled, false otheriwse
				 */
				virtual bool isEnabled() const { return m_isEnabled; }
				
				/**
				 * Get the index
				 * @retval long The local menu index of this item
				 */
				virtual long getIndex() const { return m_index; }

				/**
				 * Get the normal image
				 * @retval IImage* The normal image
				 */
				virtual IImage *getNormalImage() const { return m_normalImage; }

				/**
				 * Get the rollover image
				 * @retval IImage* The rollover image
				 */
				virtual IImage *getRolloverImage() const { return m_rolloverImage; }

				/**
				 * Set the text colour
				 * @param colour The colour to use for text
				 */
				void setTextColour(const CAlphaColour &colour) { m_textColour = colour; }

				/**
				 * Get the text colour
				 * @retval const CAlphaColour & The colour to be used to draw this menu item
				 */
				const CAlphaColour &getTextColour() const { return m_textColour; }

				/**
				 * Should the colour be overwridden? If this is true, then m_textColour is used for drawing, else default colours of the menu window are used
				 * @see CMenuWindow
				 * @see CMenuItemDisplay
				 * @param override If true text is drawn with text col
				 */
				void overrideTextColour(const bool override) { m_overrideTextColour = override; }

				/**
				 * Chck if text colours should be overridden
				 * @retval bool True if text colour can be overridden, false otherwise
				 */
				bool overrideTextColour() const { return m_overrideTextColour; }

//	===========================================================================

			protected:

//	===========================================================================

				CString m_text;					/**< The display string */
				long m_id;						/**< The identifier of the menu */
				long m_index;					/**< The index within this menu */
				EMenuItemTypes m_type;			/**< The type of menu item */
				bool m_isEnabled;				/**< Is the item enabled? */
				IImage *m_normalImage;			/**< The normal image */
				IImage *m_rolloverImage;		/**< The image to use on rollover.. */
				CAlphaColour m_textColour;		/**< Colour of the text to draw */
				bool m_overrideTextColour;		/**< Is m_textCOlour valid? */
			};
		}
	}
}
#endif	// End of CMenuItem.hpp